#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/time.h>
#include <termios.h>
#include <errno.h>
#include <stdlib.h>
#include <signal.h>
#include <ctype.h>
#include <getopt.h>

#include <lcdp_main.h>
#include <lcdp_lcd.h>
#include <lcdp_mode.h>
//#include <lcdp_lock.h>
#include <lcdp_sockets.h>

char build_date[] = "17-02-1999";
int Quit = 0;

int num_display_lines = 4;
char version[] = "0.3.5";
int run_daemon=0;
int hold_screen = 0;

void freeze_screen();

void freeze_screen() {
  hold_screen = 1;
  //fprintf(stderr, "signal USR1 caught\n");
}

void unfreeze_screen();

void unfreeze_screen() {
  hold_screen = 2;
  //fprintf(stderr, "signal USR2 caught\n");
  //mode functions are responsible for reinitializing hold_screen
}

/*
  Mode List:
    See below...  (default_sequence[])
*/

typedef struct mode {
  int name;
  int num_times;
  int delay_time;
} mode;


mode default_sequence[] =  
{
  { WITH_CPU, 		32, 	1, },	// CPU
  { WITH_MEMORY, 	32, 	2, }, 	// Memory
  { WITH_TIME, 		32, 	4, },	// Time/Date
  { WITH_DISKSTATS, 	32,  	1, },	// Disk stats
  { WITH_CREDITS,	4,   	16,},	// About (credits)
  { 1, 			0,   	0, },   // Modes after this line will not 
					// be run by default...
                    			// ... all non-default modes must be 
					// in here!
                    			// ... they will not show up otherwise.
  { WITH_OLD_TIME, 	8,   	4, },	// Old Timescreen
  { WITH_XLOAD, 	1,   	32,},	// X-load (load histogram)
  { WITH_UPTIME, 	8,   	4, },	// Old Uptime Screen
  { WITH_BATTERY, 	32,  	1, },	// Battery Status
  { WITH_CPU_GRAPH, 	32,  	1, },	// CPU histogram Graph
  { WITH_TEST_SCREEN,   32, 	1, },   // Test screen for debugging purposes
  {0, 0, 0}
};

void HelpScreen();
void exit_program(int val);
void main_loop(mode *sequence);

#define MAX_CONFIG_FILE_LENGTH 256
#define MAX_SEQUENCE 256
// 1/8th second is a single time unit...
#define TIME_UNIT 125000

char device[256] = "/dev/lcd";
/* char cfgfile[256] = "/etc/lcdproc.cf"; */
mode sequence[MAX_SEQUENCE];
char driver[256] = "MtxOrb";
int i, k, c;
int already_running = 0;
int contrast = 50;
int tmp;
int option_index;
int num_opts = 0;
int device_flag = 0;
int driver_flag = 0; 
int contrast_flag = 0;
int xload_flag = 0;
int cpu_graph_flag = 0;
int old_time_flag = 0;
int battery_flag = 0;
int uptime_flag = 0;

int process_options(int len, char **opts) {
  struct option long_options[] = {
    {"with-cpu", no_argument, NULL, WITH_CPU},
    {"with-memory", no_argument, NULL, WITH_MEMORY},
    {"with-time", no_argument, NULL, WITH_TIME},
    {"with-xload", no_argument, NULL, WITH_XLOAD},
    {"with-diskstats", no_argument, NULL, WITH_DISKSTATS},
    {"with-credits", no_argument, NULL, WITH_CREDITS},
    {"with-old-time", no_argument, NULL, WITH_OLD_TIME},
    {"with-uptime", no_argument, NULL, WITH_UPTIME},
    {"with-battery", no_argument, NULL, WITH_BATTERY},
    {"with-cpu-graph", no_argument, NULL, WITH_CPU_GRAPH},
    //{"with-test-screen", no_argument, NULL, WITH_TEST_SCREEN},
    {"device", required_argument, NULL, DEVICE},
    {"driver", required_argument, NULL, DRIVER},
    {"contrast", required_argument, NULL, CONTRAST},
    {"display-lines", required_argument, NULL, DISPLAY_LINES},
    {"config-file", required_argument, NULL, CONFIG_FILE},
    {"version", no_argument, NULL, MYVERSION},
    {"help", no_argument, NULL, HELP},
    {0, 0, 0, 0}
    };

/* Standard command line option processing loop */  
  i = 0;
  while ((c = getopt_long (len, opts, "c:d:f:l:v", 
	long_options, &option_index)) != -1)
    {
    switch (c)
      {
      case HELP:	
	HelpScreen();
	break;

      case MYVERSION:     // fall through
      case 'v':
	printf("lcdproc %s \n", version);
	exit(EXIT_SUCCESS);
	break;

      case CONTRAST: 	// fall through
      case 'c':
	if (contrast_flag == 0) {
          contrast = atoi(optarg);
        }
	contrast_flag++;
        break;

      case DEVICE:	// fall through
      case 'd':
	if (device_flag == 0) {
          strcpy(device, optarg);
        } 
	device_flag++;
        break;

      case DRIVER:	// fall through
      case 'l':
	if (driver_flag == 0) {
          strcpy(driver, optarg);
	  if (strcmp(driver, "curses") == 0) 
		run_daemon = 1;
	  if (strcmp(driver, "text")   == 0)
		run_daemon = 1;
	  if (strcmp(driver, "debug")  == 0)
		run_daemon = 1;
	}
	driver_flag++;
        break;

      case DISPLAY_LINES:
	num_display_lines = atoi(optarg);

        if (xload_flag && num_display_lines < 4) {
	   fprintf(stderr, "lcdproc: mode `--with-xload' only implemented on 20x4 displays\n"); exit(EXIT_FAILURE);
	}
	if (cpu_graph_flag && num_display_lines < 4) {
	   fprintf(stderr, "lcdproc: mode `--with-cpu-graph' only implemented on 20x4 displays\n"); exit(EXIT_FAILURE);
        }
	if (old_time_flag && num_display_lines < 4) {
	   fprintf(stderr, "lcdproc: mode `--with-old-time' deprecated - only implemented on 20x4 displays\n"); exit(EXIT_FAILURE);
	}
	if (battery_flag && num_display_lines < 4) {
	  fprintf(stderr, "lcdproc: mode `--with-battery' only implemented on 20x4 displays\n"); exit(EXIT_FAILURE); 
	}
	if (uptime_flag && num_display_lines < 4) {
	  fprintf(stderr, "lcdproc: mode `--with-uptime' deprecated - only implemented on 20x4 displays\n"); exit(EXIT_FAILURE);
	}
	break;

      case CONFIG_FILE:	// fall through
      case 'f':
	break;

      case WITH_XLOAD:
	xload_flag = 1;
	if (num_display_lines < 4) {
	   fprintf(stderr, "lcdproc: mode `--with-xload' only implemented on 20x4 displays\n"); exit(EXIT_FAILURE);
        }
      case WITH_CPU_GRAPH:  
        cpu_graph_flag = 1;
	if (num_display_lines < 4) {
	   fprintf(stderr, "lcdproc: mode `--with-cpu-graph' only implemented on 20x4 displays\n"); exit(EXIT_FAILURE);
        }
      case WITH_OLD_TIME:
	old_time_flag = 1;
	if (num_display_lines < 4) {
	   fprintf(stderr, "lcdproc: mode `--with-old-time' deprecated - only implemented on 20x4 displays\n"); exit(EXIT_FAILURE);
	}
      case WITH_BATTERY:
	battery_flag = 1;
	if (num_display_lines < 4) {
	  fprintf(stderr, "lcdproc: mode `--with-battery' only implemented on 20x4 displays\n"); exit(EXIT_FAILURE); 
	}
      case WITH_UPTIME:
	uptime_flag = 1;
	if (num_display_lines < 4) {
	  fprintf(stderr, "lcdproc: mode `--with-uptime' deprecated - only implemented on 20x4 displays\n"); exit(EXIT_FAILURE);
	}
      case WITH_CPU:           /* these cases fall through... */
      case WITH_MEMORY:
      case WITH_TIME:
      case WITH_DISKSTATS: 
      case WITH_CREDITS:
      //case WITH_TEST_SCREEN:
	sequence[i].name = c;
        for(tmp=0, k=0; default_sequence[k].name; k++)
        {
          if(sequence[i].name == default_sequence[k].name)
          {
             memcpy(&sequence[i], &default_sequence[k], sizeof(mode));
             tmp=1;
             break;
          }
        }
        i++;
    	num_opts++;
	break;

      case '?':
	exit(EXIT_FAILURE);
	break;

      default:
        abort ();
      }
    }
  if (num_opts == 0) {
  	memcpy(sequence, default_sequence, sizeof(default_sequence));
  }
  return(0);
}

void HelpScreen()
{
  printf("lcdproc, %s\n", version);
  printf("Copyright (c) 1999 ip23 gmbh\n");
  printf("Copyright (c) 1998 William Ferrell and Scott Scriven\n");
  printf("lcdproc is freely redistributable under the terms of the GNU Public License\n\n");
  printf("Usage:\n");
  printf("\t  --help \t\t - print this message\n");
  printf("\t  -v\n");
  printf("\t  --version \t\t - print the version of lcdproc being used\n");
  printf("\n");
  printf("\tGeneral configuration options:\n");
  printf("\t  -c\n");
  printf("\t  -contrast <contrast> \t - sets the screen contrast (0 - 255)\n");
  printf("\t  -f\n");
  printf("\t  --config-file <file> \t - use <file> for command line opts\n"); 
  printf("\t  -d\n");
  printf("\t  --device <device> \t - is what the lcd display is hooked up to\n");
  printf("\t\t\t\t   (/dev/cua0?)\n");
  printf("\t  --display-lines <num>\t - set the number of display lines (2,4)\n");
  printf("\t  -l <driver> \n");
  printf("\t  --driver <driver> \t - use <driver> as an output driver:\n");
  printf("\t\t\t\t   MtxOrb, PICanLCD, HD44780, curses, text,\n");
  printf("\t\t\t\t   debug\n");
  printf("\n");
  printf("\tMode display options:\n");
  printf("\t  --with-cpu \t\t - CPU usage\n");
  printf("\t  --with-memory \t - memory usage\n");
  printf("\t  --with-time \t\t - date, time, uptime, OS version\n");
  printf("\t  --with-xload \t\t - Xload (only on 20x4)\n");
  printf("\t  --with-diskstats \t - disk statistics\n");
  printf("\t  --with-credits \t - display credits\n");
  printf("\t  --with-old-time \t - old date/time (deprecated, only on 20x4)\n");
  printf("\t  --with-uptime \t - system uptime (deprecated, only on 20x4)\n");
  printf("\t  --with-battery \t - battery stats (only on 20x4)\n");
  printf("\t  --with-cpu-graph \t - CPU graph like Xload (only on 20x4)\n\n");
  printf("Report bugs to Lorand Bruhacs <bruhacs@inm.de>\n");  

  printf("\n");
  exit(EXIT_SUCCESS);
}


///////////////////////////////////////////////////////////////////
// Called upon TERM and INTR signals...
//
void exit_program(int val)
{
  Quit = 1;
  //unlock();  // Not needed any more...
  CloseAllConnections();
  goodbye_screen(0);
  lcd.flush();
  lcd.backlight(1);
  lcd.close();
  mode_close();
  exit(0);
}

char** read_configfile(char* filename); 

char** read_configfile(char* filename) {
	FILE foo;
}

int main(int argc, char **argv)
{
/*
  struct option long_options[] = {
    {"with-cpu", no_argument, NULL, WITH_CPU},
    {"with-memory", no_argument, NULL, WITH_MEMORY},
    {"with-time", no_argument, NULL, WITH_TIME},
    {"with-xload", no_argument, NULL, WITH_XLOAD},
    {"with-diskstats", no_argument, NULL, WITH_DISKSTATS},
    {"with-credits", no_argument, NULL, WITH_CREDITS},
    {"with-old-time", no_argument, NULL, WITH_OLD_TIME},
    {"with-uptime", no_argument, NULL, WITH_UPTIME},
    {"with-battery", no_argument, NULL, WITH_BATTERY},
    {"with-cpu-graph", no_argument, NULL, WITH_CPU_GRAPH},
    //{"with-test-screen", no_argument, NULL, WITH_TEST_SCREEN},
    {"device", required_argument, NULL, DEVICE},
    {"driver", required_argument, NULL, DRIVER},
    {"contrast", required_argument, NULL, CONTRAST},
    {"display-lines", required_argument, NULL, DISPLAY_LINES},
    {"config-file", required_argument, NULL, CONFIG_FILE},
    {"version", no_argument, NULL, MYVERSION},
    {"help", no_argument, NULL, HELP},
    {0, 0, 0, 0}
    };
*/
  int child;

  memset(sequence, 0, sizeof(mode) * MAX_SEQUENCE);

  // Ctrl-C will cause a clean exit...
  signal(SIGINT, exit_program);
  // and "kill"...
  signal(SIGTERM, exit_program);
  // and "kill -HUP" (hangup)...
  signal(SIGHUP, exit_program);
  // hold screen on USR1
  signal(SIGUSR1, freeze_screen);
  // unfreeze screen on USR2
  signal(SIGUSR2, unfreeze_screen);
  
  // Check to see if we are already running...
//  already_running = CheckForLock();
  already_running = PingLCDport();
  if(already_running < 0)
    {
      printf("Error checking for another lcdproc.\n");
      return -1;
    }
  
  
  // Communicate with the previously running LCDproc
  if(already_running > 0)
  {
    // "already_running" holds the pid of the LCDproc we want to talk to...

    // Do the command line  ("lcd pet status", or "lcd contrast 50")
    // And stuff...
    // ...Then exit
    printf("Detected another lcdproc.  (%i)  Exiting...\n", already_running);
    // Remove me ^^^^
    return 0;
  }

  tmp = StartSocketServer();
  if (tmp <= 0)
    {
      printf("Error starting socket server.\n");
      return 0;
    }

// This function processes command line options and initializes 
// the display sequence 
  process_options(argc, argv);
  
  // Init the com port
  if(lcd_init(device, driver) < 1) 
  {
    printf("Cannot initialize %s.\n", device);
    exit(1);
  }
  mode_init();
  lcd.contrast(contrast);

// Fork and run in the background as a daemon

   if (run_daemon=0) {
		// dont't fork
   } else {  	// fork
   	if ((child = fork()) != 0) exit(0);             /* PARENT EXITS */
   	setsid();                                       /* RELEASE TTY */
   }

  main_loop(sequence);
  
  // Clean up
  exit_program(0);
  return 0;
}


///////////////////////////////////////////////////////////////////
// Main program loop...
//
void main_loop(mode *sequence)
{
  int i, j, k;
  int Quit=0;
  int status=0;
  int timer=0;

  int blink=0;
  int hold=0;
  int heartbeat=1;

  // Main loop
  // Run whatever screen we want, then wait.  Woo-hoo!
  for(i=0; !Quit; )
  {
    timer=0;
    for(j=0; hold || (j<sequence[i].num_times && !Quit); j++)
    {
      switch(sequence[i].name)
      {
        case WITH_CPU:          status = cpu_screen(j); break;
        case WITH_MEMORY:	status = mem_screen(j); break;
        case WITH_TIME:		status = time_screen(j); break;
        case WITH_XLOAD:	status = xload_screen(j); break;
        case WITH_DISKSTATS: 	status = disk_screen(j); break;
        case WITH_CREDITS:	status = credit_screen(j); break;
        case WITH_OLD_TIME:	status = clock_screen(j); break;
        case WITH_UPTIME:	status = uptime_screen(j); break;
        case WITH_BATTERY:	status = battery_screen(j); break;
        case WITH_CPU_GRAPH:    status = cpu_graph_screen(j); break;
      //case WITH_TEST_SCREEN:	status = test_screen(j); break;
        default: 		status = dumbass_screen(j); break;
      }

      
     
      
      for(k=0; k<sequence[i].delay_time; k++)
      {
        usleep(TIME_UNIT);  timer++;
        // Modify lcd status here...  (blinking, for example)
        switch(status)
        {
          case BLINK_ON : blink=1; break;
          case BLINK_OFF: blink=0; lcd.backlight(1); break;
          case BACKLIGHT_OFF: blink=0; lcd.backlight(0); break;
          case BACKLIGHT_ON : blink=0; lcd.backlight(1); break;
	  case HOLD_SCREEN : hold=1; break;
	  case CONTINUE    : hold=0; break;
        }
	status=0;
	
        if(blink) lcd.backlight(! ((timer&15) == 15));

	if(heartbeat)
	{
	  // Set this to pulsate like a real heart beat...
	  // (binary is fun...  :)
	  lcd.icon(timer%4, 0);
	  lcd.chr(lcd.wid, 1, 0);
	}

	lcd.flush();
	
	PollSockets();
      }
    }
    i++;
    /* only display default modes */
    if(sequence[i].name < 2) i=0;
  }
}
