/* Driver module for Hitachi HD44780 based LCD display operated via 
 * the PIC-an-LCD controller.
 *
 *
 * Date   Who  What   Why
 * 990106 erd  1.0    Released
 * 990107 erd  1.1    Added flag to indicate screen needs total refreshing
 *
 * Portions of this file are...
 * Copyright (c) 1999, Ethan Dicks
 *
 * except for the parts I stole from...
 *
 * Copyright (c) 1998 Richard Rognlie       GNU Public License  
 *                    <rrognlie@gamerz.net>
 *
 * Large quantities of this code lifted (nearly verbatim) from
 * the lcd4.c module of lcdtext.  Copyright (C) 1997 Matthias Prinke
 * <m.prinke@trashcan.mcnet.de> and covered by GNU's GPL.
 * In particular, this program is free software and comes WITHOUT
 * ANY WARRANTY.
 *
 * Matthias stole (er, adapted) the code from the package lcdtime by
 * Benjamin Tse (blt@mundil.cs.mu.oz.au), August/October 1995
 * which uses the LCD-controller's 8 bit-mode.
 * References: port.H             by <damianf@wpi.edu>
 *             Data Sheet LTN211, Philips
 *             PIC-an-LCD manual, version 0.9c, August, 1997, by Dale Wheat
 *             Various FAQs and TXTs about Hitachi's LCD Controller HD44780 -
 *                www.paranoia.com/~filipg is a good starting point  ???   
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <termios.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <ctype.h>
#include "lcd.h"
#include "picanlcd.h"
#include "drv_base.h"

#define pic_write write

// PIC-an-LCD ASCII commands
//#define PICANLCD_HOME '\001'
//#define PICANLCD_CLR  '\014'
#define PICANLCD_LED  '\025'
#define PICANLCD_LCDI '\021'
#define PICANLCD_LCDD '\022'

// HD44780 raw codes
#define HD44780_CLEAR   0x01
#define HD44780_HOME    0x02
#define HD44780_CURSOFF 0x0c
#define HD44780_CGADR   0x40
#define HD44780_DDADR   0x80

// File descriptor for output stream to PIC-an-LCD port
static int fd;

// Flag to indicate that frame buffer needs to be entirely copied to LCD
static int redraw=1;

static int lcd_x=0;
static int lcd_y=0;

static void picanlcd_linewrap(int on);
static void picanlcd_autoscroll(int on);

// Buffer to store last framebuf for comparison
static char last[20*4+1];

//
// In order to directly control the HD44780, it is necessary to prefix
// the raw instruction or raw data with a control character.  These
// two-byte strings can be used as a way to send either the escape byte
// alone (with a one byte write) or to send the escape byte and the data
// together (with a two byte write)
static char send_lcd_inst[2];
static char send_lcd_data[2];

/////////////////////////////////////////////////////////////////
// Opens com port and sets baud correctly...
//
// Input: char *device - device name to open
//
// Output: returns more than 1 for success, 1 or less for failure
//
// Affects: fd - file descriptor of output stream set to valid stream or -1
//       
//
int picanlcd_init(char *device) 
{
  struct termios portset;

  register int i;

  // Set up io port correctly, and open it...
  fd = open(device, O_RDWR | O_NOCTTY | O_NDELAY);
  if (fd == -1) {
    fprintf(stderr, "picnlcd_init: failed (%s)\n", strerror(errno));
    return -1;
  } 

  // Get serial device parameters
  tcgetattr(fd, &portset);
  // This is necessary in Linux, but does not exist in irix.
#ifdef LINUX
  cfmakeraw(&portset);
#endif
  // Set port speed to 9600 baud
  cfsetospeed(&portset, B9600);
  // Set TCSANOW mode of serial device
  tcsetattr(fd, TCSANOW, &portset);

  // Set up for emitting raw data (to print programmable characters)
  send_lcd_data[0]=PICANLCD_LCDD;
  send_lcd_inst[0]=PICANLCD_LCDI;

  //set lcd on (4), cursor_on (2), and cursor_blink(1)
  send_lcd_inst[1]=HD44780_CURSOFF;  // LCD on, cursor off, no blink
  pic_write(fd, send_lcd_inst, 2);

  // Clear LCD
  send_lcd_inst[1]=HD44780_CLEAR;  // Clear Display
  pic_write(fd, send_lcd_inst, 2);

  // Set display-specific stuff..
  //picanlcd_linewrap(1);
  //picanlcd_autoscroll(1);

  // Make sure the frame buffer is there...  
  if (!lcd.framebuf) 
    lcd.framebuf = (unsigned char *) malloc(lcd.wid * lcd.hgt);

  if (!lcd.framebuf) {
    lcd.close();
    return -1;
  }

  // Clear framebuffer
  lcd.clear();

  // Clear the comparison buffer
  for (i=0; i<=20*4; i++)
    last[i]=0;


  // Set the functions the driver supports...

  lcd.init = picanlcd_init;
  lcd.close= picanlcd_close;
  lcd.clear= picanlcd_clear;
  lcd.flush_box =  picanlcd_flush_box;
  lcd.contrast =   picanlcd_contrast;
  //lcd.backlight =  picanlcd_backlight;
  lcd.set_char =   picanlcd_set_char;
  lcd.icon =       picanlcd_icon;
  lcd.init_vbar =  picanlcd_init_vbar;
  lcd.init_hbar =  picanlcd_init_hbar;
  lcd.init_num =   picanlcd_init_num;
  lcd.vbar =       picanlcd_vbar;
  lcd.hbar =       picanlcd_hbar;
  lcd.num =        picanlcd_num;
  lcd.draw_frame = picanlcd_draw_frame;

  // Tell the world how to scribble on the display
  return 999;
}

/////////////////////////////////////////////////////////////////
// Clean-up
//
void picanlcd_close() 
{
  close (fd);
  drv_base_close();
}

/////////////////////////////////////////////////////////////////
// Clear LCD
//
void picanlcd_clear() 
{
  redraw=1;          // Mark down that the screen needs to be refreshed.
  drv_base_clear();  // Clear out frame buffer
}

/////////////////////////////////////////////////////////////////
// Position Cursor
//
// A four-line display does not address the character cells linearly.
// The first data byte is the start of the first row, followed by the
// *third* row, a gap, then the second and fourth rows.
//
static void picanlcd_position(int x, int y)
{

	// First, pick high or low section of data memory
	int val = x + (y%2) * 0x40;

	// Then, add an offset if this is line three or four
	if (y>=2) val += 20;

	// Send the position command
	send_lcd_inst[1]=HD44780_DDADR | val;  // Position
	pic_write(fd, send_lcd_inst, 2);

	// Mirror the cursor location for proper framebuffer updates
	lcd_x = x;
	lcd_y = y;
}

/////////////////////////////////////////////////////////////////
//
// This appears to be disused.
//
void picanlcd_flush_box(int lft, int top, int rgt, int bot)
{
  int y;
  
  fprintf(stderr, "Flush (%i,%i)-(%i,%i)\n", lft, top, rgt, bot);

//  for (y=top; y<=bot; y++) {
//    picanlcd_position(lft,y);
//    pic_write(fd, lcd.framebuf+(y*lcd.wid)+lft, rgt-lft+1);
//  }

}


/////////////////////////////////////////////////////////////////
// Changes screen contrast (0-255; 140 seems good)
//
void picanlcd_contrast(int contrast) 
{
}

/////////////////////////////////////////////////////////////////
// Sets the backlight on or off -- can be done quickly for
// an intermediate brightness...
//
void picanlcd_backlight(int on)
{
}


/////////////////////////////////////////////////////////////////
// Toggle the built-in linewrapping feature
//
static void picanlcd_linewrap(int on)
{
}

/////////////////////////////////////////////////////////////////
// Toggle the built-in automatic scrolling feature
//
static void picanlcd_autoscroll(int on)
{
//	HD44780_senddata(0,4 | on * 2);
}


/////////////////////////////////////////////////////////////////
// Sets up for vertical bars.  Call before lcd.vbar()
//
void picanlcd_init_vbar() 
{
  char a[] = {
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    1,1,1,1,1,
  };
  char b[] = {
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    1,1,1,1,1,
    1,1,1,1,1,
  };
  char c[] = {
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
  };
  char d[] = {
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
  };
  char e[] = {
    0,0,0,0,0,
    0,0,0,0,0,
    0,0,0,0,0,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
  };
  char f[] = {
    0,0,0,0,0,
    0,0,0,0,0,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
  };
  char g[] = {
    0,0,0,0,0,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
    1,1,1,1,1,
  };

  lcd.set_char(1,a);
  lcd.set_char(2,b);
  lcd.set_char(3,c);
  lcd.set_char(4,d);
  lcd.set_char(5,e);
  lcd.set_char(6,f);
  lcd.set_char(7,g);
  
}

/////////////////////////////////////////////////////////////////
// Inits horizontal bars...
//
void picanlcd_init_hbar() 
{

  char a[] = {
    1,0,0,0,0,
    1,0,0,0,0,
    1,0,0,0,0,
    1,0,0,0,0,
    1,0,0,0,0,
    1,0,0,0,0,
    1,0,0,0,0,
    1,0,0,0,0,
  };
  char b[] = {
    1,1,0,0,0,
    1,1,0,0,0,
    1,1,0,0,0,
    1,1,0,0,0,
    1,1,0,0,0,
    1,1,0,0,0,
    1,1,0,0,0,
    1,1,0,0,0,
  };
  char c[] = {
    1,1,1,0,0,
    1,1,1,0,0,
    1,1,1,0,0,
    1,1,1,0,0,
    1,1,1,0,0,
    1,1,1,0,0,
    1,1,1,0,0,
    1,1,1,0,0,
  };
  char d[] = {
    1,1,1,1,0,
    1,1,1,1,0,
    1,1,1,1,0,
    1,1,1,1,0,
    1,1,1,1,0,
    1,1,1,1,0,
    1,1,1,1,0,
    1,1,1,1,0,
  };

  lcd.set_char(1,a);
  lcd.set_char(2,b);
  lcd.set_char(3,c);
  lcd.set_char(4,d);

}

/////////////////////////////////////////////////////////////////
// Draws a vertical bar...
//
void picanlcd_vbar(int x, int len) 
{
	char map[9] = {32, 1, 2, 3, 4, 5, 6, 7, 255 };

	int y;
	for(y=lcd.hgt; y > 0 && len>0; y--) {
		if (len >= lcd.cellhgt)
			lcd.chr(x, y, 255);
		else
			lcd.chr(x, y, map[len]);

		len -= lcd.cellhgt;
	}
  
}

/////////////////////////////////////////////////////////////////
// Draws a horizontal bar to the right.
//
void picanlcd_hbar(int x, int y, int len) 
{
	char map[6] = { 32, 1, 2, 3, 4, 255  };

	for (; x<lcd.wid && len>0; x++) {
		if (len >= lcd.cellwid)
			lcd.chr(x,y,255);
		else
			lcd.chr(x, y, map[len]);

		len -= lcd.cellwid;

	}
}


/////////////////////////////////////////////////////////////////
// Sets up for big numbers.
//
void picanlcd_init_num() 
{
	char out[3];
	sprintf(out, "%cn", 254);
	//write(fd, out, 2);
}


/////////////////////////////////////////////////////////////////
// Writes a big number.
//
void picanlcd_num(int x, int num) 
{
  char out[5];
  sprintf(out, "%c#%c%c", 254, x, num);
  //write(fd, out, 4);
}


/////////////////////////////////////////////////////////////////
// Sets a custom character from 0-7...
//
// For input, values > 0 mean "on" and values <= 0 are "off".
//
// The input is just an array of characters...
//
void picanlcd_set_char(int n, char *dat)
{
	int row, col;
	int letter;

	if(n < 0 || n > 7) return;
	if(!dat) return;

	// Set chargen address
        send_lcd_inst[1]= (HD44780_CGADR | (n*8));
        pic_write(fd, send_lcd_inst, 2);

	// Load up character data
	for(row=0; row<lcd.cellhgt; row++) {
		letter = 0;
		for(col=0; col<lcd.cellwid; col++) {
			letter <<= 1;
			letter |= (dat[(row*lcd.cellwid) + col] > 0);
		}
		send_lcd_data[1]=letter;
		pic_write(fd, send_lcd_data, 2);
	}
}


void picanlcd_icon(int which, char dest)
{
  char icons[3][5*8] = {
   {
     1,1,1,1,1,  // Empty Heart
     1,0,1,0,1,
     0,0,0,0,0,
     0,0,0,0,0,
     0,0,0,0,0,
     1,0,0,0,1,
     1,1,0,1,1,
     1,1,1,1,1,
   },   

   {
     1,1,1,1,1,  // Filled Heart
     1,0,1,0,1,
     0,1,0,1,0,
     0,1,1,1,0,
     0,1,1,1,0,
     1,0,1,0,1,
     1,1,0,1,1,
     1,1,1,1,1,
   },
   
   {
     0,0,0,0,0,  // Ellipsis
     0,0,0,0,0,
     0,0,0,0,0,
     0,0,0,0,0,
     0,0,0,0,0,
     0,0,0,0,0,
     0,0,0,0,0,
     1,0,1,0,1,
   },
   
  };

  lcd.set_char(dest, &icons[which][0]);
}


/////////////////////////////////////////////////////////////
// Blasts a single frame onscreen, to the lcd...
//
// Input is a character array, sized lcd.wid*lcd.hgt
//
void picanlcd_draw_frame(char *dat)
{
  int x,y,i;
  register char *framep, *lastp;

  // If no framebuffer, don't try to write it out
  if (!dat) return;
 
  // For each line in the framebuffer... 
  for (y=0,framep=lcd.framebuf,lastp=last; y<=3; y++) {
    // Position the cursor...
    picanlcd_position(0,y);

    // Then for each character, check to see if it's the same as last time
    for (x=0; x<lcd.wid; x++,framep++,lastp++) {
      if ((*lastp != *framep) || redraw) {
        picanlcd_position(x,y);
	// Escape non-printing characters (programmable chars)
        if (*framep < 0x20)  // ASCII space
	  pic_write(fd, send_lcd_data, 1);
	// Must be new data.  Update the LCD display
        pic_write(fd, framep, 1);
      }
    }
  }

  // Remember what this frame looked like for next time
  for (i=0; i<=20*4; i++)
    last[i]=*(lcd.framebuf+i);

  redraw=0;
}
